package com.lwby.marketing.att.novel.handle;

import com.alicp.jetcache.anno.CacheRefresh;
import com.alicp.jetcache.anno.CacheType;
import com.alicp.jetcache.anno.Cached;
import com.lwby.marketing.att.AttributionStatus;
import com.lwby.marketing.att.novel.AttributionType;
import com.lwby.marketing.att.novel.NovelUniversalProcess;
import com.lwby.marketing.flow.NodeSwitchFlow;
import com.lwby.marketing.po.CrossCallback;
import com.lwby.marketing.util.DateTimUtils;
import com.lwby.marketing.vo.NovelAction;
import com.lwby.marketing.vo.CrossPlatformAccount;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

@Component("novel_cross")
public class CrossPlatformAttributionFlow extends NodeSwitchFlow<NovelAction> {
    @Resource
    JdbcTemplate lwbyJdbcTemplate;
    @Resource
    NovelUniversalProcess up;

    private static final Logger NOVEL_SYS_LOG   = LoggerFactory.getLogger("novel.sys");

    @Override
    public boolean checked(NovelAction action) {
        //获取跨平台帐号配置
        CrossPlatformAccount cpa = getCrossPlatformMedia().get(action.getMediaName());
        //判断是否配置跨平台归因，如果没有则继续执行下一个计划归因处理器
        if(Objects.isNull(cpa) || (!cpa.getAccount().contains("0") && !cpa.getAccount().contains(action.getAdvertiserId()))
            || (cpa.getOldAccount().contains("0") || cpa.getOldAccount().contains(action.getAdvertiserId()))){
            NOVEL_SYS_LOG.info("CrossPlatformAttributionFlow checked false,platformId={},advertiserId={},media={},deviceId={},channelId={}",action.getPlatformId()
            ,action.getAdvertiserId(),action.getMediaName(),action.getDeviceId(),action.getChannelId());
            return false;
        }
        action.setCpa(cpa);
        return true;
    }

    @Override
    public void process(NovelAction action) {
        Integer sprDedu = up.getAppChannel(action.getPlatformId(), action.getChannelId());

        //等于空 或 100直接回传
        if (sprDedu == null || sprDedu == 100) {
//          callbackService.registerCallback(action);
            action.getMedia().notify(action);
            up.notifyResult(action, AttributionStatus.ACTIVE_CALLBACK);
            up.set(up.getFirstCheckerKey(action),60 * 60 * 24,"1"); //每天扣量过一次，不在重复走
            NOVEL_SYS_LOG.info("CrossPlatformAttributionFlow process0 sprDeduChannel 100,platformId={},channelId={}",action.getPlatformId(),action.getChannelId());
            return;
        }

        //总数
        String channelTotal = up.getTotalCountKey(AttributionType.CROSS_PLATFORM, action.getPlatformId(), action.getChannelId(), sprDedu, action.getCurrentDateStr());
        //回传
        String channelCallback =  up.getCallbackCountKey(AttributionType.CROSS_PLATFORM, action.getPlatformId(), action.getChannelId(), sprDedu, action.getCurrentDateStr());

        long channelTotalCount = up.incrby(channelTotal, 0, 60 * 60 * 24);
        long channelCallbackCount =  up.incrby(channelCallback, 0, 60 * 60 * 24);

        up.incrby(channelTotal,1);
        if(isCallback(action,getUserType(action),sprDedu,channelTotalCount,channelCallbackCount)){
            up.incrby(channelCallback,1);
            //callbackService.registerCallback(action);
            action.getMedia().notify(action);
            up.notifyResult(action,AttributionStatus.ACTIVE_CALLBACK);
            NOVEL_SYS_LOG.info(
                    "CrossPlatformAttributionFlow.process0.deduction doing dynamic, platformId = {}, channel = {}, sprDedu = {}, channelTotalCount = {}, channelCallbackCount = {}, v = {}",
                    action.getPlatformId(), action.getChannelId(), sprDedu, channelTotalCount, channelCallbackCount, 1);
        }else{
            up.notifyResult(action,AttributionStatus.NORMAL_DEDUCTION_CALLBACK);
            NOVEL_SYS_LOG.info(
                    "CrossPlatformAttributionFlow.process0.deduction doing dynamic, platformId = {}, channel = {}, sprDedu = {}, channelTotalCount = {}, channelCallbackCount = {}, v = {}",
                    action.getPlatformId(), action.getChannelId(), sprDedu, channelTotalCount, channelCallbackCount, 0);
        }
        up.set(up.getFirstCheckerKey(action),60 * 60 * 24,"1"); //每天扣量过一次，不在重复走
    }


    /**
     * 是否回传
     * @param action
     * @param userType
     * @param sprDedu
     * @param channelTotalCount
     * @param channelCallbackCount
     * @return
     */
    private boolean isCallback(NovelAction action, UserType userType, Integer sprDedu, long channelTotalCount, long channelCallbackCount) {
        if (userType == UserType.NEW) {
            // 纯新用户逻辑（回传）
            return true;
        } else if (userType == UserType.INTERMEDIATE) {
            // 中间用户逻辑（扣量）
            return false;
        } else {
            // 老用户逻辑
            if (channelTotalCount == 0) {
                //首次随机
                return ThreadLocalRandom.current().nextInt(1, 3) == 1;
            } else {
                //计算
                BigDecimal divide = BigDecimal.valueOf(channelCallbackCount).divide(BigDecimal.valueOf(channelTotalCount), 4, RoundingMode.HALF_UP);
                //扣量
                double percent = (double) Math.round(sprDedu * 100 / 100.0) / 100;
                return divide.compareTo(BigDecimal.valueOf(percent).setScale(4, RoundingMode.HALF_UP)) != 1;
            }
        }
    }

    /**
     * 取得用户类型（新、中、老）
     * @param action
     * @return
     */
    private UserType getUserType(NovelAction action) {
        UserType userType = UserType.OLD;
        long daysSinceRegistration = 0;

        if (DateTimUtils.isCurrentDayTime(action.getRegistrationDate())) {
            // 当日注册新用户
            daysSinceRegistration = DateTimUtils.calculateDaysBetweenDates(getEarliestRegistrationTimeByDeviceId(action.getDeviceId(),action.getPlatformId()), new Date());

            int spanCheckMaxDay = action.getCpa().getSpanCheckMaxDay();

            if (daysSinceRegistration < spanCheckMaxDay) {
                userType = UserType.INTERMEDIATE;
            } else if (daysSinceRegistration>=spanCheckMaxDay && daysSinceRegistration<=150) {
                userType = UserType.NEW;
            }
        } else {
            // 非当日注册用户
            daysSinceRegistration = DateTimUtils.calculateDaysBetweenDates(action.getRegistrationDate(), new Date());
            if (daysSinceRegistration<150) {
                userType = UserType.INTERMEDIATE;
            }
        }

        return userType;
    }

    /**
     * 获取跨包配置
     * @return
     */
    @Cached(name="cross_platform_account", cacheType = CacheType.LOCAL)
    @CacheRefresh(refresh = 300)
    public Map<String,CrossPlatformAccount> getCrossPlatformMedia(){
        RowMapper<CrossCallback> rowMapper = BeanPropertyRowMapper.newInstance(CrossCallback.class);
        List<CrossCallback> ls = lwbyJdbcTemplate.query("select span_check_max_day,new_account,old_account,media_name from cross_callback", rowMapper);

        return ls.stream()
                .collect(Collectors.toMap(CrossCallback::getMediaName, CrossPlatformAccount::new));
    }

    /**
     * 获取设备ID的最早注册时间
     * @return
     */
    public Date getEarliestRegistrationTimeByDeviceId(String deviceId,Integer platformId){
        Date date = lwbyJdbcTemplate.queryForObject(String.format("select max(registration_date) from user_profiles where platform_id != %d and device_id = '%s'",platformId,deviceId),Date.class);
        return Objects.isNull(date)?new Date():date;
    }

    enum UserType{
        NEW,INTERMEDIATE,OLD;
    }
}