package com.pika.proxy.conf;

import java.net.URI;
import java.util.BitSet;
import java.util.List;
import java.util.stream.IntStream;
import java.util.zip.CRC32;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

@Component
public class PikaShardLoad {

        private static final int SLOT_SIZE = 1024;
        private final Logger logger = LoggerFactory.getLogger(PikaShardLoad.class);
        private final JedisPool[] dict = new JedisPool[SLOT_SIZE];
        @Resource
        private PikaProxyConfig config;


        public JedisPool getPool(byte[] key){
                CRC32 ck = new CRC32();
                ck.update(key);
                return dict[(int) (ck.getValue()%SLOT_SIZE)];
        }

        @PostConstruct
        private void initShards() {
                checkpoint();
                List<Shard> shards = config.getShards();
                int maxWaitMillis = config.getMaxWaitMillis();
                int timeBetweenEvictionRunsMillis = config.getTimeBetweenEvictionRunsMillis();
                int minEvictableIdleTimeMillis = config.getMinEvictableIdleTimeMillis();
                int timeout = config.getTimeout();
                int maxIdle = config.getMaxIdle();
                int maxTotal = config.getMaxTotal();

                shards.forEach(e->{
                        String uri = e.getUri();
                        int slotEnd = e.getSlot_end();
                        int slotBegin = e.getSlot_begin();

                        JedisPoolConfig poolConfig = new JedisPoolConfig();
                        poolConfig.setMaxTotal(maxTotal);
                        poolConfig.setMaxIdle(maxIdle);
                        poolConfig.setMaxWaitMillis(maxWaitMillis);
                        poolConfig.setTestWhileIdle(true);
                        poolConfig.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
                        poolConfig.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);

                        JedisPool jp = new JedisPool(poolConfig, URI.create(uri), timeout);

                        IntStream.range(slotBegin,slotEnd+1).forEach(i -> {
                                dict[i] = jp;
                        });
                });

        }

        private void checkpoint() {
                BitSet bitSet = new BitSet(SLOT_SIZE);
                config.getShards().forEach(e -> {
                        IntStream.range(e.getSlot_begin(), e.getSlot_end() + 1).forEach(i -> {
                                if (bitSet.get(i)) {
                                        throw new RuntimeException("slot " + i + " is conflict used");
                                }
                                bitSet.set(i);
                        });
                });

                if (bitSet.cardinality() != SLOT_SIZE) {
                        throw new RuntimeException("slot is not full");
                }
        }


        private static final ThreadLocal<CRC32> crc32 = new ThreadLocal<CRC32>() {
                @Override
                protected CRC32 initialValue() {
                        return new CRC32();
                }
        };
}
