package com.pika.proxy.handler;

import com.pika.proxy.exceptions.PikaException;
import com.pika.proxy.result.ErrorResult;
import com.pika.proxy.result.InlineResult;
import com.pika.proxy.result.Result;
import com.pika.proxy.server.Server;
import com.pika.proxy.conf.PikaShardLoad;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

public abstract class CmdHandler {

        public Result exec(String key, PikaCmdHandler.IHandle handle) {
                return exec(getBytes(key), handle);
        }

        public Result exec(byte[] kb, PikaCmdHandler.IHandle handle) {
                JedisPool jp = getRedisShard().getPool(kb);
                if (jp == null) {
                        return ErrorResult.SHARD_DISABLE_RESP;
                }

                try (Jedis jedis = jp.getResource()) {
                        return handle.handle(kb,jedis);
                }
        }

        public abstract PikaShardLoad getRedisShard();

        public interface IHandle {
                Result handle(byte[] key, Jedis jedis);
        }

        protected byte[] getBytes(String key) {
                return key.getBytes(StandardCharsets.UTF_8);
        }

        protected String getString(byte[] data) {
                return new String(data, StandardCharsets.UTF_8);
        }

        public Result ping() {
                return new InlineResult("PONG");
        }
        
        public abstract Result get(String key) throws PikaException;

        public abstract Result set(String key, byte[] value) throws PikaException;

        public abstract Result mget(String key, byte[][] keys) throws PikaException;

        public abstract Result setEx(String key, int second, byte[] value) throws PikaException;

        public abstract Result setNx(String key, byte[] value) throws PikaException;

        public abstract Result getSet(String key, byte[] value) throws PikaException;

        public abstract Result del(String key) throws PikaException;

        public abstract Result exists(String key) throws PikaException;

        public abstract Result expire(String key, int value) throws PikaException;

        public abstract Result expireAt(String key, int value) throws PikaException;

        public abstract Result ttl(String key) throws PikaException;

        public abstract Result decr(String key) throws PikaException;

        public abstract Result decrBy(String key, long value) throws PikaException;

        public abstract Result incr(String key) throws PikaException;

        public abstract Result incrBy(String key, long value) throws PikaException;

        public abstract Result hSet(String key, byte[] field, byte[] value) throws PikaException;

        public abstract Result hMSet(String key, byte[][] value) throws PikaException;

        public abstract Result hGet(String key, byte[] field) throws PikaException;

        public abstract Result hMGet(String key, byte[][] field) throws PikaException;

        public abstract Result hGetAll(String key) throws PikaException;

        public abstract Result hDel(String key, byte[][] field) throws PikaException;

        public abstract Result hExists(String key, byte[] field) throws PikaException;

        public abstract Result hIncrBy(String key, byte[] field, long value) throws PikaException;

        public abstract Result hKeys(String key) throws PikaException;

        public abstract Result hLen(String key) throws PikaException;

        public abstract Result hSetNx(String key, byte[] field, byte[] value) throws PikaException;

        public abstract Result hVals(String key) throws PikaException;

        public abstract Result lPush(String key, byte[][] value) throws PikaException;

        public abstract Result rPush(String key, byte[][] value) throws PikaException;

        public abstract Result lPop(String key) throws PikaException;

        public abstract Result rPop(String key) throws PikaException;

        public abstract Result bLPop(String key, int timeout) throws PikaException;

        public abstract Result bRPop(String key, int timeout) throws PikaException;

        public abstract Result lLen(String key) throws PikaException;

        public abstract Result lRange(String key, long start, long end) throws PikaException;

        public abstract Result lRem(String key, long count, byte[] value) throws PikaException;

        public abstract Result lSet(String key, long index, byte[] value) throws PikaException;

        public abstract Result lTrim(String key, long start, long end) throws PikaException;

        public abstract Result sAdd(String key, byte[][] value) throws PikaException;

        public abstract Result sCard(String key) throws PikaException;

        public abstract Result sisMember(String key, byte[] value) throws PikaException;

        public abstract Result sMembers(String key) throws PikaException;

        public abstract Result sPop(String key) throws PikaException;

        public abstract Result sRem(String key, byte[][] value) throws PikaException;

        public abstract Result zAdd(String key, byte[][] value) throws PikaException;

        public abstract Result zCard(String key) throws PikaException;

        public abstract Result zCount(String key, byte[] min, byte[] max) throws PikaException;

        public abstract Result zIncrBy(String key, double score, byte[] value) throws PikaException;

        public abstract Result zRange(String key, long stat, long stop) throws PikaException;

        public abstract Result zRangeByScore(String key, byte[][] value) throws PikaException;

        public abstract Result zRank(String key, byte[] value) throws PikaException;

        public abstract Result zRem(String key, byte[][] value) throws PikaException;

        public abstract Result zRemRangeByRank(String key, long start, long stop) throws PikaException;

        public abstract Result zRemRangeByScore(String key, byte[] min, byte[] max) throws PikaException;

        public abstract Result zRevRange(String key, long start, long stop) throws PikaException;

        public abstract Result zRevRangeByScore(String key, byte[][] value) throws PikaException;

        public abstract Result zRevRank(String key, byte[] value) throws PikaException;

        public abstract Result zScore(String key, byte[] value) throws PikaException;
}
