package com.pika.proxy.handler;

import com.google.common.collect.Maps;
import com.pika.proxy.conf.PikaShardLoad;

import com.pika.proxy.exceptions.PikaException;
import com.pika.proxy.result.*;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.annotation.Resource;
import java.util.Map;

@Component
public class PikaCmdHandler extends CmdHandler {
        @Resource
        private PikaShardLoad redisShard;

        @Override
        public PikaShardLoad getRedisShard() {
                return redisShard;
        }

        @Override
        public Result get(String key) {
                return exec(key, (bk, jedis) -> new BulkResult(jedis.get(bk)));
        }

        @Override
        public Result set(String key, byte[] value) {
                return exec(key,(bk,jedis)->new StatusResult(jedis.set(bk,value)));
        }

        public Result mget(String key, byte[][] keys) {
                Result[] rs = new Result[keys.length + 1];
                rs[0] = exec(key,(bk,jedis)->new BulkResult(jedis.get(bk)));

                for(int i=0;i<keys.length;i++){
                        rs[i + 1] = exec(keys[i],(bk,jedis)->new BulkResult(jedis.get(bk)));
                }

                return new MultiBulkResult(rs);
        }

        @Override
        public Result setEx(String key, int second, byte[] value) {
                return exec(key, (bk, jedis) -> new StatusResult(jedis.setex(bk, second, value)));
        }

        @Override
        public Result setNx(String key, byte[] value){
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.setnx(bk, value)));
        }

        @Override
        public Result getSet(String key, byte[] value) {
                return exec(key, (bk, jedis) -> new BulkResult(jedis.getSet(bk, value)));
        }

        @Override
        public Result del(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.del(bk)));

        }

        @Override
        public Result exists(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.exists(bk)));
        }

        @Override
        public Result expire(String key, int value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.expire(bk, value)));
        }

        @Override
        public Result expireAt(String key, int value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.expireAt(bk, value)));
        }

        @Override
        public Result ttl(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.ttl(bk)));
        }

        @Override
        public Result decr(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.decr(bk)));
        }

        @Override
        public Result decrBy(String key, long value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.decrBy(bk, value)));
        }

        @Override
        public Result incr(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.incr(bk)));
        }

        @Override
        public Result incrBy(String key, long value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.incrBy(bk, value)));
        }

        @Override
        public Result hSet(String key, byte[] field, byte[] value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hset(bk, field, value)));
        }

        @Override
        public Result hMSet(String key, byte[][] value) {
                return exec(key, (bk, jedis) -> {
                        Map<byte[], byte[]> map = Maps.newHashMap();
                        for (int i = 0; i < value.length;) {
                                map.put(value[i++], value[i++]);
                        }

                        return new StatusResult(jedis.hmset(bk, map));
                });
        }

        @Override
        public Result hGet(String key, byte[] field) {
                return exec(key, (bk, jedis) -> new BulkResult(jedis.hget(bk, field)));
        }

        @Override
        public Result hMGet(String key, byte[][] field) {
                return exec(key, (bk, jedis) -> new MultiBulkResult(jedis.hmget(bk, field)));
        }

        @Override
        public Result hGetAll(String key) {
                return exec(key, (bk, jedis) -> new MultiBulkResult(jedis.hgetAll(bk)));
        }

        @Override
        public Result hDel(String key, byte[][] field) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hdel(bk, field)));
        }

        @Override
        public Result hExists(String key, byte[] field) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hexists(bk, field)));
        }

        @Override
        public Result hIncrBy(String key, byte[] field, long value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hincrBy(bk, field, value)));
        }

        @Override
        public Result hKeys(String key) {
                return exec(key, (bk, jedis) -> new MultiBulkResult(jedis.hkeys(bk)));
        }

        @Override
        public Result hLen(String key) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hlen(bk)));
        }

        @Override
        public Result hSetNx(String key, byte[] field, byte[] value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.hsetnx(bk, field, value)));
        }

        @Override
        public Result hVals(String key) {
                return exec(key, (bk, jedis) -> new MultiBulkResult(jedis.hvals(bk)));
        }

        @Override
        public Result lPush(String key, byte[][] vlaue) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.lpush(bk, vlaue)));
        }

        @Override
        public Result rPush(String key, byte[][] value) {
                return exec(key, (bk, jedis) -> new IntegerResult(jedis.rpush(bk, value)));
        }

        @Override
        public Result lPop(String key) {
                return exec(key, (bk, jedis) ->  new BulkResult(jedis.lpop(bk)));
        }

        @Override
        public Result rPop(String key) {
                return exec(key, (bk, jedis) ->  new BulkResult(jedis.rpop(bk)));
        }

        @Override
        public Result bLPop(String key, int timeout) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.blpop(timeout, bk)));
        }

        @Override
        public Result bRPop(String key, int timeout) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.brpop(timeout, bk)));
        }

        @Override
        public Result lLen(String key) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.llen(bk)));
        }

        @Override
        public Result lRange(String key, long start, long end) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.lrange(bk, start, end)));
        }

        @Override
        public Result lRem(String key, long count, byte[] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.lrem(bk, count, value)));
        }

        @Override
        public Result lSet(String key, long index, byte[] value) {
                return exec(key, (bk, jedis) ->  new StatusResult(jedis.lset(bk, index, value)));
        }

        @Override
        public Result lTrim(String key, long start, long end) {
                return exec(key, (bk, jedis) ->  new StatusResult(jedis.ltrim(bk, start, end)));
        }

        @Override
        public Result sAdd(String key, byte[][] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.sadd(bk, value)));
        }

        @Override
        public Result sCard(String key) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.scard(bk)));
        }

        @Override
        public Result sisMember(String key, byte[] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.sismember(bk, value)));
        }

        @Override
        public Result sMembers(String key) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.smembers(bk)));
        }

        @Override
        public Result sPop(String key) {
                return exec(key, (bk, jedis) ->  new BulkResult(jedis.spop(bk)));
        }

        @Override
        public Result sRem(String key, byte[][] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.srem(bk, value)));
        }

        @Override
        public Result zAdd(String key, byte[][] value) {
                return exec(key, (bk, jedis) -> {
                        if (value.length == 2) {
                                return new IntegerResult(jedis.zadd(bk, Double.parseDouble(getString(value[0])), value[1]));
                        } else {
                                Map<byte[], Double> map = Maps.newHashMap();
                                for (int i = 0; i < value.length;) {
                                        map.put(value[i++], Double.parseDouble(getString(value[i++])));
                                }
                                return new IntegerResult(jedis.zadd(bk, map));
                        }
                });
        }

        @Override
        public Result zCard(String key) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zcard(bk)));
        }

        @Override
        public Result zCount(String key, byte[] min, byte[] max) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zcount(bk, min, max)));
        }

        @Override
        public Result zIncrBy(String key, double score, byte[] value) {
                return exec(key, (bk, jedis) ->  new BulkResult(jedis.zincrby(bk, score, value)));
        }

        @Override
        public Result zRange(String key, long start, long stop) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.zrange(bk, start, stop)));
        }

        @Override
        public Result zRangeByScore(String key, byte[][] value) {
                return exec(key, (bk, jedis) -> {
                        if (value.length == 2) {
                                return new MultiBulkResult(jedis.zrangeByScore(bk, value[0], value[1]));
                        } else if (value.length == 5) {
                                return new MultiBulkResult(jedis.zrangeByScore(bk, value[0], value[1], Integer.parseInt(getString(value[3])), Integer.parseInt(getString(value[4]))));
                        } else {
                                return new ErrorResult("wrong number of args for 'zrangebyscore' command");
                        }
                });
        }

        @Override
        public Result zRank(String key, byte[] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zrank(bk, value)));
        }

        @Override
        public Result zRem(String key, byte[][] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zrem(bk, value)));
        }

        @Override
        public Result zRemRangeByRank(String key, long start, long stop) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zremrangeByRank(bk, start, stop)));
        }

        @Override
        public Result zRemRangeByScore(String key, byte[] min, byte[] max) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zremrangeByScore(bk, min, max)));
        }

        @Override
        public Result zRevRange(String key, long start, long stop) {
                return exec(key, (bk, jedis) ->  new MultiBulkResult(jedis.zrevrange(bk, start, stop)));
        }

        @Override
        public Result zRevRangeByScore(String key, byte[][] value) {
                return exec(key, (bk, jedis) -> {
                        if (value.length == 2) {
                                return new MultiBulkResult(jedis.zrevrangeByScore(bk, value[0], value[1]));
                        } else if (value.length == 5) {
                                return new MultiBulkResult(jedis.zrevrangeByScore(bk, value[0], value[1], Integer.parseInt(getString(value[3])), Integer.parseInt(getString(value[4]))));
                        } else {
                                return new ErrorResult("wrong number of args for 'zrangebyscore' command");
                        }
                });
        }

        @Override
        public Result zRevRank(String key, byte[] value) {
                return exec(key, (bk, jedis) ->  new IntegerResult(jedis.zrevrank(bk, value)));
        }

        @Override
        public Result zScore(String key, byte[] value) {
                return exec(key, (bk, jedis) -> new BulkResult(jedis.zscore(bk, value)));
        }
}
