package com.pika.proxy.server;

import com.pika.proxy.server.codec.PikaCmd;
import com.pika.proxy.handler.CmdHandler;
import com.pika.proxy.invoke.CmdInvoke;
import com.pika.proxy.invoke.PikaCmdInvokeImpl;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import com.pika.proxy.result.ErrorResult;
import com.pika.proxy.result.Result;
import com.pika.proxy.result.StatusResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class ServerHandler extends SimpleChannelInboundHandler<PikaCmd> {

        private final Logger logger = LoggerFactory.getLogger(ServerHandler.class);
        private final ThreadPoolExecutor executor;
        private final Map<String, CmdInvoke> methods;

        public ServerHandler(int minThreads, int maxThreads, CmdHandler handler) {
                executor = new ThreadPoolExecutor(minThreads, maxThreads, 30, TimeUnit.MINUTES, new LinkedBlockingQueue(Short.MAX_VALUE));
                methods = new HashMap<>();
                Class<? extends CmdHandler> clazz = handler.getClass();
                for (final Method method : clazz.getMethods()) {
                        Class<?>[] types = method.getParameterTypes();
                        CmdInvoke invoke = new PikaCmdInvokeImpl(types, method, handler);

                        methods.put(method.getName(), invoke);
                        methods.put(method.getName().toUpperCase(), invoke);
                        methods.put(method.getName().toLowerCase(), invoke);
                }
        }


        @Override
        protected void channelRead0(ChannelHandlerContext ctx, PikaCmd cmd) {
                executor.execute(() -> {
                        try {
                                handler(ctx, cmd);
                        } catch (Exception e) {
                                ctx.writeAndFlush(new ErrorResult(e));
                                e.printStackTrace();
                        }
                });
        }

        private void handler(ChannelHandlerContext ctx, final PikaCmd cmd) throws Exception {
                String name = new String(cmd.getName(), StandardCharsets.UTF_8);
                CmdInvoke invoke = methods.get(name);
                Result<?> resp;
                if (invoke == null) {
                        resp = new ErrorResult("unknown command '" + name + "'");
                } else {
                        resp = invoke.execute(name, cmd);
                }
                if (resp == StatusResult.QUIT) {
                        ctx.close();
                } else {
                        if (resp == null) {
                                resp = ErrorResult.NYI_RESP;
                        }
                        ctx.writeAndFlush(resp);
                }
        }

        @Override
        public void channelReadComplete(ChannelHandlerContext ctx) {
                ctx.flush();
        }

        @Override
        public void channelActive(ChannelHandlerContext ctx)  {
                logger.info("远 程服务  " + ctx.channel().remoteAddress() + "  链路建立");
        }

        @Override
        public void channelInactive(ChannelHandlerContext ctx)  {
                logger.info("远程服务  " + ctx.channel().remoteAddress() + "  链路断开");
        }

        @Override
        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
                logger.error(cause + "");
                ctx.close();
        }
}
